<?php

namespace App\Http\Controllers\AdminControllers;

use App\Http\Controllers\Controller;
use App\Services\BlogServices;
use App\Services\BlogCategoryServices;
use App\Services\BlogTagServices;
use App\Services\TagServices;
use App\Models\CommentSettings;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Str;
use Yajra\DataTables\Facades\DataTables;

class AdminBlogController extends Controller
{
    protected $blog;
    protected $blog_category;
    protected $blog_tag;

    public function __construct()
    {
        Parent::__construct();

        $this->blog = new BlogServices();
        $this->blog_category = new BlogCategoryServices();
        $this->blog_tag = new BlogTagServices();
        $this->tag = new TagServices();
    }
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index()
    {
        try {
            if (request()->ajax()) {
                $get = $this->blog->getData()->get();
                return DataTables::of($get)
                    ->addIndexColumn()
                    ->addColumn('action', function ($get) {
                        $button  = "<div class='d-flex'>
                                <a href='" . route('admin.blog.weblog.edit', $get->id) . "'>
                                    <button type='button' class='btn btn-sm btn btn-success m-1' data-id='" . $get->id . "'>" . __('Edit') . "</button>
                                </a>
                                <a href='javascript:void(0)' title='Delete' class='deleteRow'>
                                    <button type='button' class='btn btn-sm btn btn-danger m-1'>" . __('Delete') . "</button>
                                    <form action=" . route('admin.blog.weblog.destroy', $get->id) . " method='post' class='deleteForm'>
                                        " . csrf_field() . "
                                        <input type='hidden' name='_method' value='delete'>
                                    </form>
                                </a>
                            </div>";
                        return $button;
                    })
                    ->addColumn('bg_image', function ($get) {
                        return imageShow($get->bg_image, 'Image', '', 'width="80"');
                    })
                    ->addColumn('feature_image', function ($get) {
                        return imageShow($get->feature_img, 'Image', '', 'width="80"');
                    })
                    ->addColumn('category', function ($get) {
                        return ($get->category != null ? $get->category->name : null);
                    })
                    ->addColumn('is_popular', function ($get) {
                        $popular  = "<label class='switch'>
                        <input type='checkbox'" . ($get->is_popular ? 'checked' : '') . " id='" . $get->id . "' class='blogIsPopularBtn'>
                        <span class='slider round'></span>
                    </label>";
                        return $popular;
                    })
                    ->addColumn('description', function ($get) {
                        return Str::limit(strip_tags($get->description), 60, '...');
                    })
                    ->addColumn('is_feature', function ($get) {
                        $feature  = "<label class='switch'>
                        <input type='checkbox' " . ($get->is_featured ? 'checked' : '') . " id='" . $get->id . "' class='blogIsFeaturedBtn'>
                        <span class='slider round'></span>
                    </label>";
                        return $feature;
                    })
                    ->rawColumns(['action', 'bg_image', 'feature_image', 'category', 'is_popular', 'is_feature'])->make(true);
            }
            return view('backend.pages.blogs.blog', [
                'title' => 'Blog',
            ]);
        } catch (\Exception $th) {
            return redirect()->back()->with('error', $th->getMessage());
        }
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        try {
            $blog = null;
            return view('backend.pages.blogs.blog-form', [
                'title' => 'Blog',
                'blog' => $blog,
                'tags' => $this->tag->getData()->get(),
                'blog_tags' => [],
                'categories' => $this->blog_category->getData()->get(),
            ]);
        } catch (\Exception $th) {
            return redirect()->back()->with('error', $th->getMessage());
        }
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {

        try {

            $this->validate($request, [
                'title' => 'required',
                'description' => 'required',
                'category_id' => 'required',
                'bg_image.*' => 'nullable|mimes:jpg,jpeg,png,gif',
                'feature_img.*' => 'nullable|mimes:jpg,jpeg,png,gif',
            ]);

            $data = [
                'title' => $request->title,
                'slug' => Str::slug($request->title),
                'user_id' => $request->user_id,
                'category_id' => $request->category_id,
                'description' => $request->description
            ];

            if ($request->hasFile('bg_image')) {
                $data['bg_image'] = fileUpload($request->bg_image, UPLOAD_FOLDER_BLOGS_BG_IMAGES)[0];
            }
            if ($request->hasFile('feature_img')) {
                $data['feature_img'] = fileUpload($request->feature_img, UPLOAD_FOLDER_BLOGS_FEATURES)[0];
            }

            $blog = $this->blog->store($data);


            if ($request->has('blog_tags')) {
                $nblog_tag = [];
                foreach ($request->blog_tags as $blog_tag) {
                    $nblog_tag[] =
                        [
                            'blog_id' => $blog->id,
                            'tag_id' => $blog_tag
                        ];
                }
                $this->blog_tag->getData()->insert($nblog_tag);
            }

            return $this->backWithSuccess('Blog has been created successfully', 'admin.blog.weblog.index');
        } catch (\Exception $th) {
            return redirect()->back()->with('error', $th->getMessage());
        }
    }

    /**
     * Display the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function show($id)
    {

        try {
            $blog = $this->blog->findById($id);

            if ($blog->is_popular) {
                $blog->is_popular = false;
                $blog->save();
                $msg = 'Blog popular successfully.';
                $status = true;
            } else {
                $blog->is_popular = true;
                $blog->save();
                $msg = 'Blog unpopular successfully.';
                $status = false;
            }
            return response()->json(['msg' => $msg, 'status' => $status]);
        } catch (\Exception $th) {
            return redirect()->back()->with('error', $th->getMessage());
        }
    }

    public function makeFeatured($id)
    {
        try {
            $blog = $this->blog->findById($id);

            if ($blog->is_featured) {
                $blog->is_featured = false;
                $blog->save();
                $msg = 'Blog featured successfully.';
                $status = true;
            } else {
                $blog->is_featured = true;
                $blog->save();
                $msg = 'Blog unfeatured successfully.';
                $status = false;
            }
            return response()->json(['msg' => $msg, 'status' => $status]);
        } catch (\Exception $th) {
            return redirect()->back()->with('error', $th->getMessage());
        }
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function edit($id)
    {
        try {

            $fblog = $this->blog->findById($id);
            $blog_tag_arr = $fblog->blogTags->pluck('tag_id')->toArray();


            return view('backend.pages.blogs.blog-form', [
                'title' => 'Blog',
                'blog' => $fblog,
                'tags' => $this->tag->getData()->get(),
                'blog_tags' => $blog_tag_arr,
                'categories' => $this->blog_category->getData()->get(),
                'userName' => Auth::user(),
            ]);
        } catch (\Exception $th) {
            return redirect()->back()->with('error', $th->getMessage());
        }
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, $id)
    {
        try {
            $this->validate($request, [
                'title' => 'required',
                'description' => 'required',
                'category_id' => 'required',
                'bg_image.*' => 'nullable|mimes:jpg,jpeg,png,gif',
                'feature_img.*' => 'nullable|mimes:jpg,jpeg,png,gif',
            ]);

            $blog = $this->blog->findById($id);
            DB::beginTransaction();
            $data = [
                'title' => $request->title,
                'slug' => Str::of($request->title)->slug('-'),
                'category_id' => $request->category_id,
                'description' => $request->description
            ];
            if ($request->hasFile('bg_image')) {
                $data['bg_image'] = fileUpload($request->bg_image, UPLOAD_FOLDER_BLOGS_BG_IMAGES, $blog->getAttributes()['bg_image'])[0];
            }
            if ($request->hasFile('feature_img')) {
                $data['feature_img'] = fileUpload($request->feature_img, UPLOAD_FOLDER_BLOGS_FEATURES, $blog->getAttributes()['feature_img'])[0];
            }

            $blog->update($data);

            $this->blog_tag->getData()->where('blog_id', $blog->id)->delete();
            if (!empty($request->blog_tags)) {
                $nblog_tag = [];
                foreach ($request->blog_tags as $blog_tag) {
                    $nblog_tag[] =
                        [
                            'blog_id' => $blog->id,
                            'tag_id' => $blog_tag
                        ];
                }
                $this->blog_tag->getData()->insert($nblog_tag);
            }

            DB::commit();

            return $this->backWithSuccess('Blog has been created successfully', 'admin.blog.weblog.index');
        } catch (\Exception $th) {
            DB::rollBack();
            return redirect()->back()->with('error', $th->getMessage());
        }
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function destroy($id)
    {
        try {
            $blog = $this->blog->findById($id);
            destroryFile(UPLOAD_FOLDER_BLOGS_BG_IMAGES, $blog->getAttributes()['bg_image']);
            destroryFile(UPLOAD_FOLDER_BLOGS_FEATURES, $blog->getAttributes()['feature_img']);
            $blog->delete();

            return $this->backWithSuccess('Blog has been deleted successfully');
        } catch (\Exception $th) {
            return redirect()->back()->with('error', $th->getMessage());
        }
    }

    public function commentSettingsIndex()
    {
        try {
            $title = 'Top Header';
            $commentSetting = CommentSettings::first();
            return view('backend.pages.blogs.comment', compact('title', 'commentSetting'));
        } catch (\Throwable $th) {
            return $this->backWithError($th->getMessage());
        }
    }

    public function commentSettingsStore(Request $request)
    {
        try {
            if ($request->has('show')) {
                $request->validate([
                    'code' => 'required',
                    'url' => 'nullable',
                ], [
                    'code' => 'Comment code is required.',
                    'url' => 'url is required.'
                ]);
            }

            $commentSetting = CommentSettings::first() ? CommentSettings::first() : new CommentSettings();
            $commentSetting->show = $request->has('show');
            $commentSetting->code = $request->code;
            $commentSetting->url = $request->url;
            $commentSetting->save();
            return $this->backWithSuccess('Comment settings updated successfully.');
        } catch (\Throwable $th) {
            return $this->backWithError($th->getMessage());
        }
    }
}
