<?php

namespace App\Http\Controllers\AdminControllers;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Services\DiscountService;
use App\Services\DiscountSubService;
use App\Services\PackageService;
use App\Services\HotelService;
use Carbon\Carbon;
use Yajra\DataTables\Facades\DataTables;

class AdminDiscountController extends Controller
{
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    protected $discount;
    protected $package;
    protected $hotel;
    protected $discount_sub_price;
    public function __construct()
    {
        parent::__construct();
        $this->discount = new DiscountService;
        $this->package = new PackageService;
        $this->hotel = new HotelService;
        $this->discount_sub_price = new DiscountSubService;
    }
    public function index()
    {
        try {
            if (request()->ajax()) {
                $get = $this->discount->getData()->orderBy('start_date_time', 'desc');
                return DataTables::of($get)
                    ->addIndexColumn()
                    ->addColumn('action', function ($get) {
                        return "<div class='d-flex'>
                                <a href='" . route('admin.discounts.edit', $get->id) . "'>
                                    <button type='button' class='btn btn-sm btn btn-success m-1' data-id='" . $get->id . "'>" . __('Edit') . "</button>
                                </a>
                                <a href='javascript:void(0)' title='Delete' class='deleteRow'>
                                    <button type='button' class='btn btn-sm btn btn-danger m-1'>" . __('Delete') . "</button>
                                    <form action=" . route('admin.discounts.destroy', $get->id) . " method='post' class='deleteForm'>
                                        " . csrf_field() . "
                                        <input type='hidden' name='_method' value='delete'>
                                    </form>
                                </a>
                            </div>";
                    })
                    ->addColumn('images', function ($get) {
                        return imageShow($get->images, 'Image', '', 'width="100"');
                    })
                    ->addColumn('discount_type', function ($get) {
                        return discountType($get->discount_type);
                    })
                    ->addColumn('amount', function ($get) {
                        return ($get->discount_type == DISCOUNT_FIXED ? currencyFormat($get->amount) : $get->amount . '%');
                    })
                    ->addColumn('status', function ($get) {
                        return statusActiveDeactive($get->status);
                    })
                    ->addColumn('related', function ($get) {
                        if ($get->hotelDiscount) {
                            return __('Hotel:') . $get->hotelDiscount->count();
                        }

                        if ($get->hotelDiscount) {
                            return __('Package:') . $get->packageDiscount->count();
                        }
                    })
                    ->rawColumns(['action', 'images', 'discount_type', 'amount', 'status', 'related'])
                    ->make(true);
            }
            return view('backend.pages.discount.table', [
                'title' => 'Discount'
            ]);
        } catch (\Exception $th) {
            return redirect()->back()->with('error', $th->getMessage());
        }
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        try {
            return view('backend.pages.discount.form', [
                'title' => 'Discount',
                'discount' => null,
                'package' => $this->package->getData()->get(),
                'hotel' => $this->hotel->getData()->get(),
                'discount_hotel' => [],
                'discount_package' => [],
                'start_end_date' => ''
            ]);
        } catch (\Exception $th) {
            return redirect()->back()->with('error', $th->getMessage());
        }
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {

        $request->validate([
            'title' => 'required|string|max:255',
            'discount_type' => 'required|string|max:255',
            'package' => ['nullable', 'max:255'],
            'hotel' => ['nullable', 'max:255'],
            'details' => 'required|string|max:255',
            'amount' => 'required|string|max:255|regex:/^([0-9.]+)$/',
            'start_end_date' => 'required|string|max:255',
            'images.*' => 'nullable|image|mimes:jpeg,png,jpg,gif,svg|max:2048',
        ]);
        try {
            $start_end_date = explode(' - ', $request->start_end_date);
            $start_date_time = date('Y-m-d H:i:s', strtotime($start_end_date[0]));
            $end_date_time = date('Y-m-d H:i:s', strtotime($start_end_date[1]));
            $this->discount->store([
                'title' => $request->title,
                'package' => $request->package,
                'hotel' => $request->hotel,
                'discount_type' => $request->discount_type,
                'details' => $request->details,
                'start_date_time' => $start_date_time,
                'amount' => $request->amount,
                'end_date_time' => $end_date_time,
                'status' => (isset($request->status) ? ACTIVE_STATUS : DEACTIVE_STATUS),
                'images' => $request->images
            ]);
            return $this->backWithSuccess('Discount has been created successfully', 'admin.discounts.index');
        } catch (\Throwable $th) {
            return redirect()->back()->with('error', $th->getMessage());
        }
    }

    /**
     * Display the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function show($id)
    {
        //
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function edit($id)
    {
        try {
            $discount = $this->discount->findById($id);
            $start_end_date = date('m-d-Y H:i A', strtotime($discount->start_date_time)) . ' - ' . date('m-d-Y H:i A', strtotime($discount->end_date_time));

            $discount_hotel = (($discount->discountSubPrice) ? $discount->discountSubPrice->pluck('hotel_id')->toArray() : []);
            $discount_package = (($discount->discountSubPrice) ? $discount->discountSubPrice->pluck('package_id')->toArray() : []);

            return view('backend.pages.discount.form', [
                'title' => 'Discount',
                'discount' => $this->discount->findById($id),
                'package' => $this->package->getData()->get(),
                'hotel' => $this->hotel->getData()->get(),
                'discount_hotel' => $discount_hotel,
                'discount_package' => $discount_package,
                'start_end_date' => $start_end_date,
            ]);
        } catch (\Exception $th) {
            return redirect()->back()->with('error', $th->getMessage());
        }
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, $id)
    {
        $request->validate([
            'title' => 'required|string|max:255',
            'discount_type' => 'required|string|max:255',
            'details' => 'required|string|max:255',
            'amount' => 'required|string|max:255|regex:/^([0-9.]+)$/',
            'start_end_date' => 'required|string|max:255',
            'status' => 'nullable|string|max:255',
        ]);

        try {
            $start_end_date = explode(' - ', $request->start_end_date);
            $start_date_time = date('Y-m-d H:i:s', strtotime($start_end_date[0]));
            $end_date_time = date('Y-m-d H:i:s', strtotime($start_end_date[1]));

            $data = [
                'title' => $request->title,
                'discount_type' => $request->discount_type,
                'details' => $request->details,
                'start_date_time' => $start_date_time,
                'amount' => $request->amount,
                'end_date_time' => $end_date_time,
                'status' => (isset($request->status) ? ACTIVE_STATUS : DEACTIVE_STATUS),
                'images' => $request->images
            ];
            $sub_data = [
                'package' => $request->package,
                'hotel' => $request->hotel,
            ];
            $this->discount->update($data, $sub_data, $id);
            return $this->backWithSuccess('Discount has been created successfully', 'admin.discounts.index');
        } catch (\Throwable $th) {
            return redirect()->back()->with('error', $th->getMessage());
        }
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function destroy($id)
    {
        try {
            $discount = $this->discount->findById($id);

            $delt = $this->discount->destroy($id);
            if ($delt) {
                unlink(public_path('upload/discount_image/' . $discount->images));
            }
            return $this->backWithSuccess('Discount Deleted successfully.');
        } catch (\Throwable $th) {
            return $this->backWithError($th->getMessage());
        }
    }
}
