<?php

namespace App\Http\Controllers\AdminControllers;

use App\Http\Controllers\Controller;
use App\Services\HotelService;
use App\Services\AccommodationService;
use App\Services\FacilityService;
use App\Services\HotelAccommodationService;
use Illuminate\Http\Request;
use Yajra\DataTables\Facades\DataTables;

class AdminHotelAccommodationController extends Controller
{
    protected $hotel;
    protected $accommodation;
    protected $facility;
    protected $hotel_accommodation;
    protected $room_images_path = 'upload/hotel-accommodation/';

    public function __construct()
    {
        Parent::__construct();
        $this->hotel = new HotelService();
        $this->accommodation = new AccommodationService();
        $this->facility = new FacilityService();
        $this->hotel_accommodation = new HotelAccommodationService();
    }
    public function index(Request $request)
    {
        try {
            if (request()->ajax()) {
                $get = $this->hotel_accommodation->getData();
                return DataTables::of($get)
                    ->addIndexColumn()
                    ->addColumn('action', function ($get) {
                        return "<div class='d-flex'>
                                    <a href='" . route('admin.hotel-accommodation.edit', $get->id) . "'>
                                    <button type='button' class='btn btn-sm btn btn-success m-1 blogCategoryEditBtn' data-id='" . $get->id . "'>" . __('Edit') . "</button>
                                </a>
                                <a href='javascript:void(0)' title='Delete' class='deleteRow'>
                                    <button type='button' class='btn btn-sm btn btn-danger m-1'>" . __('Delete') . "</button>
                                    <form action=" . route('admin.hotel-accommodation.destroy', $get->id) . " method='post' class='deleteForm'>
                                        " . csrf_field() . "
                                        <input type='hidden' name='_method' value='delete'>
                                    </form>
                                </a>
                            </div>";
                    })
                    ->addColumn('is_smoking', function ($get) {
                        return statusActiveDeactive($get->is_smoking);
                    })
                    ->addColumn('price', function ($get) {
                        return currencyFormat($get->price);
                    })
                    ->addColumn('facilities', function ($get) {
                        return ($get->facilities ? implodeFunc(', ', getFacilities($get->facilities)) : '');
                    })
                    ->addColumn('accommodation', function ($get) {
                        return (($get->accommodation) ? $get->accommodation->name : '');
                    })
                    ->addColumn('no_of_guest', function ($get) {
                        return $get->no_of_adult . __('Adult') . '<br />' . $get->no_of_children . _('Children');
                    })
                    ->filter(function ($get) use ($request) {
                        if ($request->get('hotel_id')) {
                            $get->where('hotel_id', $request->get('hotel_id'));
                        }
                        if (!empty($request->get('search'))) {
                            $get->whereHas('hotel', function ($w) use ($request) {
                                $search = $request->get('search');
                                $w->where('hotel_name', 'LIKE', "%$search%");
                            });
                        }
                    })
                    ->rawColumns(['action', 'accommodation', 'price', 'facilities', 'no_of_guest', 'is_smoking'])
                    ->make(true);
            }

            return view('backend.pages.hotel.hotel-accommodation.index', [
                'title' => 'Hotel',
                'hotel_id' => $request->hotel_id,
                'hotels' => $this->hotel->getData()->get()
            ]);
        } catch (\Throwable $e) {
            return redirect()->back()->with('error', $e->getMessage());
        }
    }
    public function create(Request $request)
    {
        try {
            $hotels = $this->hotel->getData()->get();
            $accommodations = $this->accommodation->getData()->get();

            if ($request->ajax()) {
                $hotel_accommodations = $this->hotel_accommodation->getData()->get();

                $data_html = '<option value="">' . __('Select') . '</option>';
                foreach ($hotel_accommodations as $hotel_accommodation) {
                    $data_html .= '<option value="' . $hotel_accommodation->id . '">' . $hotel_accommodation->name . '</option>';
                }
                return $data_html;
            }

            return view('backend.pages.hotel.hotel-accommodation.form', [
                'title' => 'Hotel Accommodation',
                'hotel_accommodation' => null,
                'hotel' => null,
                'hotels' => $hotels,
                'hotel_accommodation_facilities' => [],
                'facilities' => $this->facility->getData()->get(),
                'accommodations' => $accommodations,
                'room_images' => [],
                'list_hotel_accommodations' => []
            ]);
        } catch (\Throwable $e) {
            return redirect()->back()->with('error', $e->getMessage());
        }
    }

    public function hotelPriceUpdate($hotel_id)
    {
        $min_price = $this->hotel_accommodation->getData()->where('hotel_id', $hotel_id)->min('price');
        $this->hotel->update(['price' => $min_price], $hotel_id);
    }

    public function store(Request $request)
    {
        $request->validate([
            'hotel_id' => ['required', 'exists:App\Models\Hotel,id'],
            'accommodation_id' => ['required', 'exists:App\Models\Accommodation,id'],
            'no_of_room' => ['required'],
            'price' => 'required',
            'no_of_adult' => ['required', 'min:-90', 'max:90'],
            'no_of_children' => ['required', 'min:-90', 'max:90'],
            'room_size' => ['required', 'min:-180', 'max:180'],
            'room_image.*' => ['nullable', 'mimes:jpeg,jpg,png,gif'],
            'facilities.*' => ['required']
        ], [
            'hotel_id.required' => 'Hotel name is required.',
            'accommodation_id.required' => 'Accommodation is required.',
        ]);
        try {
            $data = [
                'hotel_id' => $request->hotel_id,
                'accommodation_id' => $request->accommodation_id,
                'no_of_room' => $request->no_of_room,
                'price' => $request->price,
                'no_of_adult' => $request->no_of_adult,
                'no_of_children' => $request->no_of_children,
                'room_size' => $request->room_size,
                'facilities' => ((count($request->facilities)) ? json_encode($request->facilities) : ''),
                'room_images' => $request->room_images
            ];

            $hotel_accommodation = $this->hotel_accommodation->store($data);

            $this->hotelPriceUpdate($request->hotel_id);

            return $this->backWithSuccess('Hotel Accommodation created successfully.', 'admin.hotel-accommodation.index', 'hotel_id=' . $hotel_accommodation->hotel_id);
        } catch (\Throwable $e) {
            return redirect()->back()->with('error', $e->getMessage());
        }
    }

    public function accommodationAdd($id)
    {
        try {
            $hotel = $this->hotel->findById($id);
            $list_hotel_accommodations = $hotel->hotelAccomodation;
            $hotel_accommodations = $hotel->hotelAccomodation->pluck('accommodation_id')->toArray();
            $accommodations = $this->accommodation->getData()->whereNotIn('id', $hotel_accommodations)->get();

            return view('backend.pages.hotel.hotel-accommodation.form', [
                'title' => 'Hotel Accommodation',
                'hotel_accommodation' => null,
                'hotel' => $hotel,
                'hotels' => [],
                'hotel_accommodation_facilities' => [],
                'facilities' => $this->facility->getData()->get(),
                'accommodations' => $accommodations,
                'room_images' => [],
                'list_hotel_accommodations' => $list_hotel_accommodations
            ]);
        } catch (\Throwable $e) {
            return redirect()->back()->with('error', $e->getMessage());
        }
    }

    public function imageDestroy(Request $request)
    {
        try {
            $hotel_accommodation = $this->hotel_accommodation->findById($request->hotel_accommoddation_id);
            $room_images = json_decode($hotel_accommodation->room_images, true);

            if (in_array($request->image, $room_images)) {
                destroryFile($this->room_images_path, $request->image);
                $new_room_images = array_diff($room_images, [$request->image]);
                $hotel_accommodation->update(['room_images' => $new_room_images]);

                return $this->backWithSuccess('Hotel Accommodation image destroy successfully.');
            }
        } catch (\Throwable $th) {
            return redirect()->back()->with('error', $th->getMessage());
        }
    }

    public function edit($id)
    {
        try {
            $hotel_accommodation = $this->hotel_accommodation->findById($id);
            $hotel = $this->hotel->findById($hotel_accommodation->hotel_id);
            $hotel_accommodations = $hotel->hotelAccomodation->pluck('accommodation_id')->toArray();

            return view('backend.pages.hotel.hotel-accommodation.form', [
                'title' => 'Hotel Accommodation',
                'hotel_accommodation' => $hotel_accommodation,
                'hotel' => $hotel,
                'hotels' => [],
                'facilities' => $this->facility->getData()->get(),
                'hotel_accommodation_facilities' => json_decode($hotel_accommodation->facilities, true),
                'accommodations' => $this->accommodation->getData()->whereNotIn('id', $hotel_accommodations)->get(),
                'room_images' => (($hotel_accommodation->room_images) ? json_decode($hotel_accommodation->room_images, true) : []),
                'list_hotel_accommodations' => []
            ]);
        } catch (\Throwable $e) {
            return redirect()->back()->with('error', $e->getMessage());
        }
    }

    public function update(Request $request, $id)
    {
        $request->validate([
            'hotel_id' => ['required', 'exists:App\Models\Hotel,id'],
            'accommodation_id' => ['required', 'exists:App\Models\Accommodation,id'],
            'no_of_room' => ['required'],
            'price' => 'required',
            'no_of_adult' => ['required', 'min:-90', 'max:90'],
            'no_of_children' => ['required', 'min:-90', 'max:90'],
            'room_size' => ['required', 'min:-180', 'max:180'],
            'room_image.*' => ['nullable', 'mimes:jpeg,jpg,png,gif'],
            'facilities.*' => ['required']
        ], [
            'hotel_id.required' => 'Hotel name is required.',
            'accommodation_id.required' => 'Accommodation is required.',
        ]);
        try {
            $data = [
                'hotel_id' => $request->hotel_id,
                'accommodation_id' => $request->accommodation_id,
                'no_of_room' => $request->no_of_room,
                'price' => $request->price,
                'no_of_adult' => $request->no_of_adult,
                'no_of_children' => $request->no_of_children,
                'room_size' => $request->room_size,
                'facilities' => ((count($request->facilities)) ? json_encode($request->facilities) : ''),
                'room_images' => $request->room_images,
            ];

            $this->hotel_accommodation->update($data, $request->id);
            $this->hotelPriceUpdate($request->hotel_id);

            return $this->backWithSuccess('Hotel Accommodation updated successfully.', 'admin.hotel-accommodation.index', 'hotel_id=' . $request->hotel_id);
        } catch (\Throwable $e) {
            return redirect()->back()->with('error', $e->getMessage());
        }
    }
    public function destroy($id)
    {
        try {
            $item = $this->hotel_accommodation->destroy($id);

            return $this->backWithSuccess('Hotel has been deleted successfully.', 'admin.hotel-accommodation.index', 'hotel_id=' . $item->hotel_id);
        } catch (\Throwable $th) {
            return redirect()->back()->with('error', $th->getMessage());
        }
    }
}
