<?php

namespace App\Http\Controllers\AdminControllers;

use App\Http\Controllers\Controller;
use Illuminate\Contracts\View\View;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Str;
use App\Services\PackageService;
use App\Services\PackageCategoryService;
use App\Services\PackageImageService;
use App\Services\SubPriceService;
use App\Services\PlaceService;
use Yajra\DataTables\Facades\DataTables;

class AdminPackageController extends Controller
{
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    protected $package;
    protected $category;
    protected $packageImage;
    protected $subprice;
    public function __construct()
    {
        Parent::__construct();
        $this->package = new PackageService();
        $this->category = new PackageCategoryService;
        $this->packageImage = new PackageImageService;
        $this->subprice = new SubPriceService;
        $this->place = new PlaceService;
    }
    public function index()
    {
        try {
            if (request()->ajax()) {
                $get = $this->package->getData();
                return DataTables::of($get)
                    ->addIndexColumn()
                    ->addColumn('action', function ($get) {
                        return "<div class='d-flex'>
                         <a href='" . route('admin.packages.edit', $get->id) . "'>
                                <button type='button'
                                    class='btn btn-sm btn btn-success m-1'>" . __('Edit') . "</button>
                            </a>
                                <a href='javascript:void(0)' title='Delete' class='deleteRow'>
                                    <button type='button' class='btn btn-sm btn btn-danger m-1'>" . __('Delete') . "</button>
                                    <form action=" . route('admin.packages.destroy', $get->id) . " method='post' class='deleteForm'>
                                        " . csrf_field() . "
                                        <input type='hidden' name='_method' value='delete'>
                                    </form>
                                </a>
                            </div>";
                    })
                    ->addColumn('description', function ($get) {
                        return Str::limit(strip_tags($get->description), 60, '...');
                    })
                    ->addColumn('category', function ($get) {
                        return ($get->category ? $get->category->name : '');
                    })
                    ->rawColumns(['action'])->make(true);
            }
            return view('backend.pages.packages.index', [
                'title' => 'Pakage',
                'categories' => $this->category->getData()->get(),
            ]);
        } catch (\Exception $th) {
            return redirect()->back()->with('error', $th->getMessage());
        }
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        try {
            return view('backend.pages.packages.form', [
                'title' => 'Pakage',
                'items' => $this->package->getData()->get(),
                'cats' => $this->category->getData()->get(),
                'places' => $this->place->getData()->get(),
            ]);
        } catch (\Exception $th) {
            return redirect()->back()->with('error', $th->getMessage());
        }
    }

    public function packagePriceUpdate($package_id)
    {
        $min_price = $this->subprice->getData()->where('package_id', $package_id)->min('tour_price');
        $this->package->update(['price' => $min_price], $package_id);
    }
    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        $request->validate([
            'category' => 'required|exists:package_categories,id',
            'place' => 'required|exists:places,id',
            'title' => 'required|string|max:255',
            'sub_title' => 'required|string|max:255',
            'day' => 'required|string|max:255',
            'price' => 'required|string|max:255|regex:/^([0-9]+)$/',
            'group_size' => 'required|string|max:255',
            'tour_location' => 'required|string|max:255',
            'best_time_to_go' => 'required|string|max:255',
            'tour_attractions' => 'required|string|max:255',
            'tour_availability' => 'required|string|max:255',
            'next_scheduled_tour' => 'required|string|max:255',
            'description' => 'required|string',
            'highlight' => 'required|string',
            'itineraries' => 'required|string',
            'note' => 'required|string',
            'inclusion' => 'required|string',
            'exclusion' => 'required|string',
            'bg_image.*' => 'nullable|image|mimes:jpeg,png,jpg,gif,svg|max:2048|min:1',
            'package_image.*' => 'nullable|image|mimes:jpeg,png,jpg,gif,svg|max:2048',
        ]);

        try {
            if ($request->hasFile('bg_image')) {
                $tour_map_image = fileUpload($request->bg_image, UPLOAD_FOLDER_PACKAGE_IMAGE)[0];
            }
            $data = $this->package->store([
                'category_id' => $request->category,
                'place_id' => $request->place,
                'title' => $request->title,
                'sub_title' => $request->sub_title,
                'day' => $request->day,
                'location' => $request->tour_location,
                'best_time_to_go' => $request->best_time_to_go,
                'tour_attractions' => $request->tour_attractions,
                'tour_availability' => $request->tour_availability,
                'next_scheduled_tour' => $request->next_scheduled_tour,
                'description' => $request->description,
                'tour_highlight' => $request->highlight,
                'tour_itineraries' => $request->itineraries,
                'notes' => $request->note,
                'inclusion' => $request->inclusion,
                'exclusion' => $request->exclusion,
                'tour_map_image' => (isset($tour_map_image) ? $tour_map_image : null),
            ]);
            $this->subprice->store([
                'package_id' => $data->id,
                'tour_price' => $request->price,
                'group_size' => $request->group_size,
            ]);
            $this->packagePriceUpdate($data->id);
            $i = 0;
            if ($request->hasFile('package_image')) {
                foreach ($request->package_image as $image) {
                    $package_image = fileUpload($image, UPLOAD_FOLDER_PKG_IMAGE)[0];
                    $this->packageImage->store([
                        'package_id' => $data->id,
                        'images' => $package_image,
                    ]);
                }
            }
            return $this->backWithSuccess('Package has been created successfully', 'admin.package-subprice.package', ['package_id' => $data->id]);
        } catch (\Throwable $th) {
            return redirect()->back()->with('error', $th->getMessage());
        }
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  int  $id
     * @return View|RedirectResponse
     */
    public function edit($id)
    {
        try {
            return view('backend.pages.packages.edit', [
                'title' => 'Package Edit',
                'items' => $this->package->findById($id),
                'cats' => $this->category->getData()->get(),
                'places' => $this->place->getData()->get(),
            ]);
        } catch (\Exception $th) {
            return redirect()->back()->with('error', $th->getMessage());
        }
    }
    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, $id)
    {

        $request->validate([
            'category' => 'required|string|max:255',
            'place' => 'required|string|max:255',
            'title' => 'required|string|max:255',
            'sub_title' => 'required|string|max:255',
            'day' => 'required|string|max:255',
            'tour_location' => 'required|string|max:255',
            'best_time_to_go' => 'required|string|max:255',
            'tour_attractions' => 'required|string|max:255',
            'tour_availability' => 'required|string|max:255',
            'next_scheduled_tour' => 'required|string|max:255',
            'description' => 'required|string',
            'highlight' => 'required|string',
            'itineraries' => 'required|string',
            'note' => 'required|string',
            'inclusion' => 'required|string',
            'exclusion' => 'required|string',
            'bg_image.*' => 'nullable|image|mimes:jpeg,png,jpg,gif,svg|max:2048',
            'package_image.*' => 'nullable|image|mimes:jpeg,png,jpg,gif,svg|max:2048',
        ]);
        try {
            $data = [
                'category_id' => $request->category,
                'place_id' => $request->place,
                'title' => $request->title,
                'sub_title' => $request->sub_title,
                'day' => $request->day,
                'location' => $request->tour_location,
                'best_time_to_go' => $request->best_time_to_go,
                'tour_attractions' => $request->tour_attractions,
                'tour_availability' => $request->tour_availability,
                'next_scheduled_tour' => $request->next_scheduled_tour,
                'description' => $request->description,
                'tour_highlight' => $request->highlight,
                'tour_itineraries' => $request->itineraries,
                'notes' => $request->note,
                'inclusion' => $request->inclusion,
                'exclusion' => $request->exclusion,
            ];
            if ($request->hasFile('bg_image')) {
                $file = $this->package->findById($id);
                $data['tour_map_image'] = fileUpload($request->bg_image, UPLOAD_FOLDER_PACKAGE_IMAGE, $file->getAttributes()['tour_map_image'])[0];
            }
            $this->package->update($data, $id);
            $this->package->findById($id);
            if ($request->hasFile('package_image')) {
                $i = 0;
                foreach ($request->package_image as $image) {
                    $package_image = fileUpload($image, UPLOAD_FOLDER_PKG_IMAGE)[0];
                    $this->packageImage->store([
                        'package_id' => $id,
                        'images' => $package_image,
                    ]);
                    $i = $i + 1;
                }
            }

            return $this->backWithSuccess('Package has been Updated successfully');
        } catch (\Illuminate\Validation\ValidationException $th) {
            return redirect()->back()->with('error', $th->getMessage());
        }
    }
    /**
     * Remove the specified resource from storage.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function destroy($id)
    {
        try {
            $data = $this->package->findById($id);
            $this->package->destroy($id);
            if ($data->image != null) {
                unlink(public_path("upload/package_image/" . $data->tour_map_image));
            }
            return $this->backWithSuccess('Package Deleted successfully.');
        } catch (\Throwable $th) {
            return $this->backWithError($th->getMessage());
        }
    }
}
