<?php

namespace App\Http\Controllers\AdminControllers;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Services\PackageService;
use App\Services\SubPriceService;
use Yajra\DataTables\Facades\DataTables;

class AdminPackageSubPriceController extends Controller
{
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    protected $package;
    protected $package_sub_price;
    public function __construct()
    {
        Parent::__construct();
        $this->package = new PackageService();
        $this->package_sub_price = new SubPriceService;
    }
    public function index(Request $request)
    {
        try {
            if (request()->ajax()) {
                $get = $this->package_sub_price->getData();
                return DataTables::of($get)
                    ->addIndexColumn()
                    ->addColumn('action', function ($get) {
                        return "<div class='d-flex'>
                                    <a href='" . route('admin.package-subprice.edit', $get->id) . "'>
                                    <button type='button' class='btn btn-sm btn btn-success m-1 blogCategoryEditBtn' data-id='" . $get->id . "'>" . __('Edit') . "</button>
                                </a>
                                <a href='javascript:void(0)' title='Delete' class='deleteRow'>
                                    <button type='button' class='btn btn-sm btn btn-danger m-1'>" . __('Delete') . "</button>
                                    <form action=" . route('admin.package-subprice.destroy', $get->id) . " method='post' class='deleteForm'>
                                        " . csrf_field() . "
                                        <input type='hidden' name='_method' value='delete'>
                                    </form>
                                </a>
                            </div>";
                    })
                    ->addColumn('package', function ($get) {
                        return (($get->package) ? $get->package->title : '');
                    })
                    ->filter(function ($get) use ($request) {
                        if ($request->get('package_id')) {
                            $get->where('package_id', $request->get('package_id'));
                        }
                        if (!empty($request->get('search'))) {
                            $get->whereHas('package', function ($w) use ($request) {
                                $search = $request->get('search');
                                $w->where('title', 'LIKE', "%$search%")
                                    ->orWhere('sub_title', 'LIKE', "%$search%");
                            });
                        }
                    })
                    ->rawColumns(['action', 'package'])
                    ->make(true);
            }
            return view('backend.pages.packages.subprice.table', [
                'title' => 'Sub Price',
                'packages' => $this->package->getData()->get(),
                'package_id' => $request->package_id
            ]);
        } catch (\Exception $th) {
            return redirect()->back()->with('error', $th->getMessage());
        }
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        try {
            return view('backend.pages.packages.subprice.form', [
                'title' => 'Package Sub Price',
                'packages' => $this->package->getData()->get(),
            ]);
        } catch (\Exception $th) {
            return redirect()->back()->with('error', $th->getMessage());
        }
    }


    public function packagePriceUpdate($package_id)
    {
        $min_price = $this->package_sub_price->getData()->where('package_id', $package_id)->min('tour_price');
        $this->package->update(['price' => $min_price], $package_id);
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        try {
            $this->validate($request, [
                'package' => 'required|string|max:255',
                'group_size' => 'required|string|max:255',
                'sub_price' => 'required|string|max:15',
            ]);

            $data = [
                'package_id' => $request->package,
                'group_size' => $request->group_size,
                'tour_price' => $request->sub_price,
            ];
            $this->package_sub_price->store($data);
            $this->packagePriceUpdate($request->package);

            return $this->backWithSuccess('Package Sub Price has been Stored successfully');
        } catch (\Illuminate\Validation\ValidationException $th) {
            return redirect()->back()->with('error', $th->getMessage());
        }
    }


    /**
     * Show the form for editing the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function packageSubPriceEdit($id)
    {
        try {
            return view('backend.pages.packages.subprice.edit', [
                'title' => 'Package Sub Price',
                'package' => $this->package->findById($id),
            ]);
        } catch (\Exception $th) {
            return redirect()->back()->with('error', $th->getMessage());
        }
    }
    /**
     * Show the form for editing the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function edit($id)
    {
        try {
            return view('backend.pages.packages.subprice.edit', [
                'title' => 'Package Sub Price',
                'package' => null,
                'package_sub_price' => $this->package_sub_price->findById($id),
            ]);
        } catch (\Exception $th) {
            return redirect()->back()->with('error', $th->getMessage());
        }
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function packageSubPriceupdate(Request $request, $id)
    {
        try {
            $this->validate($request, [
                'group_size.*' => 'required|string|max:255',
                'sub_price.*' => 'required|string|max:15|regex:/^([0-9.]+)$/',
            ]);
            for ($i = 0; count($request->subprice_id) > $i; $i++) {
                if ($request->subprice_id[$i] != null) {
                    $this->package_sub_price->update([
                        'package_id' => $id,
                        'group_size' => $request->group_size[$i],
                        'tour_price' => $request->sub_price[$i],
                    ], $request->subprice_id[$i]);
                } else {
                    $this->package_sub_price->store([
                        'package_id' => $id,
                        'group_size' => $request->group_size[$i],
                        'tour_price' => $request->sub_price[$i],
                    ]);
                }
            }
            $this->packagePriceUpdate($id);
            return $this->backWithSuccess('Package Sub Price has been Updated successfully', 'admin.package-departure.package.add', ['package_id' => $id]);
        } catch (\Illuminate\Validation\ValidationException $th) {
            return redirect()->back()->with('error', $th->getMessage());
        }
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, $id)
    {
        try {
            $this->validate($request, [
                'group_size' => 'required|string|max:255',
                'sub_price' => 'required|string|max:15|regex:/^([0-9.]+)$/',
            ]);
            $sub_price = $this->package_sub_price->findById($id);
            $sub_price->update([
                'group_size' => $request->group_size,
                'tour_price' => $request->sub_price,
            ]);

            $this->packagePriceUpdate($sub_price->package_id);
            return $this->backWithSuccess('Package Sub Price has been Updated successfully', 'admin.package-subprice.index', ['package_id' => $sub_price->package_id]);
        } catch (\Illuminate\Validation\ValidationException $th) {
            return redirect()->back()->with('error', $th->getMessage());
        }
    }
    /**
     * Remove the specified resource from storage.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function destroy($id)
    {
        try {
            $this->package_sub_price->destroy($id);
            return $this->backWithSuccess('Sub Price Deleted successfully.');
        } catch (\Throwable $th) {
            return $this->backWithError($th->getMessage());
        }
    }
}
