<?php

namespace App\Http\Controllers\AdminControllers;

use App\Http\Controllers\Controller;
use App\Models\Partners;
use App\Services\PartnerService;
use Illuminate\Contracts\View\View;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\Request;
use Yajra\DataTables\Facades\DataTables;

class AdminPartnerController extends Controller
{
    protected $partners;
    public function __construct()
    {
        Parent::__construct();
        $this->partners = new PartnerService;
    }
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index()
    {
        try {
            if (request()->ajax()) {
                $get = $this->partners->getData()->get();
                return DataTables::of($get)
                    ->addIndexColumn()
                    ->addColumn('action', function ($get) {
                        $button  = "<div class='d-flex'>
                                <a href='" . route('admin.partner.edit', $get->id) . "'>
                                    <button type='button' class='btn btn-sm btn btn-success m-1 blogCategoryEditBtn' data-id='" . $get->id . "'>" . __('Edit') . "</button>
                                </a>

                                <a href='javascript:void(0)' title='Delete' class='deleteRow'>
                                    <button type='button' class='btn btn-sm btn btn-danger m-1'>" . __('Delete') . "</button>
                                    <form action=" . route('admin.partner.destroy', $get->id) . " method='post' class='deleteForm'>
                                        " . csrf_field() . "
                                        <input type='hidden' name='_method' value='delete'>
                                    </form>
                                </a>
                            </div>";
                        return $button;
                    })
                    ->addColumn('image', function ($get) {
                        return imageShow($get->image, 'Image', '', 'width="80"');
                    })
                    ->addColumn('status', function ($get) {
                        $status  = "<label class='switch'>
                                        <input type='checkbox' " . ($get->status ? 'checked' : '') . " id='" . $get->id . "' class='serviceActivationBtn'>
                                        <span class='slider round'></span>
                                    </label>";
                        return $status;
                    })
                    ->addColumn('url', function ($get) {
                        return '<a href="' . $get->url . '">' . $get->url . '</a>';
                    })
                    ->rawColumns(['action', 'image', 'status', 'url'])->make(true);
            }
            return view('backend.pages.partners.index', [
                'title' => 'Partner'
            ]);
        } catch (\Throwable $e) {
            return redirect()->back()->with('error', $e->getMessage());
        }
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response|RedirectResponse
     */
    public function create()
    {
        try {
            return view('backend.pages.partners.form', [
                'title' => 'Add New Partner',
                'partner' => null
            ]);
        } catch (\Throwable $e) {
            return redirect()->back()->with('error', $e->getMessage());
        }
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        $request->validate([
            'name' => 'required|string|max:255',
            'url' => 'required|url',
            'image.*' => ['nullable', 'mimes:jpeg,png,jpg,gif']
        ], [
            'name' => 'Slider title is required.',
            'url' => 'URL title is required.',
        ]);
        try {

            $partner = new Partners();
            if ($request->hasFile('image')) {
                $partner->image = fileUpload($request->image, UPLOAD_FOLDER_PARTNERS)[0];
            }
            $partner->name = $request->name;
            $partner->url =  $request->url;
            $partner->status = $request->status == 'on' ? true : false;
            $partner->save();
            return $this->backWithSuccess('Partner created successfully.');
        } catch (\Exception $th) {
            return redirect()->back()->with('error', $th->getMessage());
        }
    }

    /**
     * Display the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function partnerShow($id)
    {
        $item = $this->partners->findById($id);
        try {
            if ($item->status) {
                $item->status = false;
                $item->save();
                $msg = 'Partner deactivation successfully.';
                $status = true;
            } else {
                $item->status = true;
                $item->save();
                $msg = 'Partner activation successfully.';
                $status = false;
            }
            return response()->json(['msg' => $msg, 'status' => $status]);
        } catch (\Exception $th) {
            return redirect()->back()->with('error', $th->getMessage());
        }
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function edit($id)
    {
        try {
            $partner = $this->partners->findById($id);
            return view('backend.pages.partners.form', [
                'title' => 'Edit Partner',
                'partner' => $partner
            ]);
        } catch (\Throwable $e) {
            return redirect()->back()->with('error', $e->getMessage());
        }
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, $id)
    {
        $request->validate([
            'name' => 'required|string|max:255',
            'url' => 'required|url',
            'image*' => ['nullable', 'mimes:jpeg,png,jpg,gif']
        ], [
            'name' => 'Slider title is required.',
            'url' => 'URL title is required.',
        ]);
        try {
            $partner = $this->partners->findById($id);
            if ($request->hasFile('image')) {
                $partner->image = fileUpload($request->image, UPLOAD_FOLDER_PARTNERS, $partner->getAttributes()['image'])[0];
            }
            $partner->name = $request->name;
            $partner->url =  $request->url;
            $partner->status = $request->status == 'on' ? true : false;
            $partner->save();
            return $this->backWithSuccess('Partner created successfully.');
        } catch (\Exception $th) {
            return redirect()->back()->with('error', $th->getMessage());
        }
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function destroy($id)
    {
        try {
            $partners = Partners::findOrfail($id);
            if ($partners->image != null) {
                destroryFile(UPLOAD_FOLDER_PARTNERS, $partners->getAttributes()['image']);
            }
            $partners->delete();
            return $this->backWithSuccess('Partner has been deleted successfully.');
        } catch (\Throwable $th) {
            return redirect()->back()->with('error', $th->getMessage());
        }
    }
}
