<?php

namespace App\Http\Controllers\GuestViewControllers;

use App\Http\Controllers\Controller;
use App\Models\PageSectionSettings;
use App\Services\BookingService;
use App\Services\HotelAccommodationService;
use App\Services\HotelBookingService;
use App\Services\HotelServices;
use App\Services\PackageBookingService;
use App\Services\PackageService;
use App\Services\Payment\StripeService;
use App\Services\PaymentMethodService;
use App\Services\SubPriceService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Validator;
use Carbon\Carbon;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Cookie;
use Illuminate\Support\Facades\URL;

class GuestBookingController extends Controller
{
    protected $hotel;
    protected $hotel_accommodation;
    protected $hotel_booking;
    protected $package_booking;
    protected $booking;
    protected $package;
    protected $package_group_price;


    public function __construct()
    {
        parent::__construct();
        $this->hotel = new HotelServices();
        $this->hotel_accommodation = new HotelAccommodationService();
        $this->hotel_booking = new HotelBookingService();
        $this->package_booking = new PackageBookingService();
        $this->booking = new BookingService();
        $this->package = new PackageService();
        $this->package_group_price = new SubPriceService();
    }

    public function bookingHotel($hotel_id, $accommodation_id)
    {
        try {
            if (!Auth::check() || (Auth::check() && Auth::user()->user_type == ADMIN_USER)) {
                Cookie::queue(COOKIE_LOGIN_AFTER_URL, route('booking.hotel', ['hotel_id' => $hotel_id, 'accommodation_id' => $accommodation_id]), 60);
                return $this->backWithWarning(__('Please login before booking hotel'), 'guest.login');
            }
            Cookie::forget(COOKIE_LOGIN_AFTER_URL);
            $title = __('Hotel Booking');
            $hotel_accommodations = $this->hotel_accommodation->getData()->where(['hotel_id' => $hotel_id])->get();
            $accommodation = $this->hotel_accommodation->getData()->where(['hotel_id' => $hotel_id, 'id' => $accommodation_id])->first();
            $discount = (($accommodation->hotel->discount) ? $accommodation->hotel->discount->discount : null);
            $images = ($accommodation->room_images) ? json_decode($accommodation->room_images, true) : [];

            return view('frontend.pages.booking.hotel', compact('title', 'hotel_accommodations', 'accommodation', 'images', 'discount'));
        } catch (\Exception $e) {
            Log::error($e->getMessage());
            return abort(404);
        }
    }

    public function bookingHotelSave(Request $request)
    {
        try {
            $validator = Validator::make($request->all(), [
                'hotel_accommodation_id' => ['required', 'exists:App\Models\HotelAccommodation,id'],
                'check_in_out' => ['required'],
                'arrival_date_time' => ['required'],
                'first_name' => ['required'], 'last_name' => ['required'],
                'email' => ['required', 'email']
            ], [
                'hotel_accommodation_id.required' => 'Hotel accommodation is required'
            ]);
            if ($validator->fails()) {
                return redirect()
                    ->back()
                    ->withErrors($validator)
                    ->withInput();
            }
            $check_in_out = explode(' - ', $request->check_in_out);
            $check_id = Carbon::createFromFormat('m-d-Y h:i A', $check_in_out[0])->format('Y-m-d H:i:s');
            $check_out = Carbon::createFromFormat('m-d-Y h:i A', $check_in_out[1])->format('Y-m-d H:i:s');
            $arrival_date_time = Carbon::createFromFormat('m-d-Y h:i A', $request->arrival_date_time)->format('Y-m-d H:i:s');

            $total_night = nightCalculation($check_id, $check_out);
            $data = [
                'hotel_id' => $request->hotel_id,
                'user_id' => Auth::id(),
                'discount_id' => $request->discount_id,
                'hotel_accommodation_id' => $request->hotel_accommodation_id,
                'check_in' => $check_id,
                'check_out' => $check_out,
                'arrival_date_time' => $arrival_date_time,
                'first_name' => $request->first_name,
                'last_name' => $request->last_name,
                'email' => $request->email,
                'total_night' => $total_night,
                'is_travel_for_work' => $request->is_travel_for_work,
                'status' => BOOKING_STATUS_PENDING
            ];

            $this->hotel_booking->store($data);
            return $this->backWithSuccess(__('Booking done. We send a email in this email account. Please check you email and confirm the confirmation link.'), 'guest.hotel.order');
        } catch (\Throwable $th) {
            Log::error($th->getMessage());
            return abort(404);
        }
    }

    public function bookingPackage($package_id, $group_id)
    {
        try {
            if (!Auth::check() || (Auth::check() && Auth::user()->user_type == ADMIN_USER)) {
                Cookie::queue(COOKIE_LOGIN_AFTER_URL, route('booking.package', ['package_id' => $package_id, 'group_id' => $group_id]), 60);
                return $this->backWithWarning(__('Please login before booking hotel'), 'guest.login');
            }
            Cookie::forget(COOKIE_LOGIN_AFTER_URL);
            $title = __('Package Booking');
            $package = $this->package->findById($package_id);
            $discount = (($package->discount) ? $package->discount->discount : null);
            $package_group_prices = $this->package_group_price->getData()->where(['package_id' => $package_id])->get();
            $group_price = $this->package_group_price->getData()->where(['package_id' => $package_id, 'id' => $group_id])->first();
            $pageContent = PageSectionSettings::where('name', 'packages_page')->first();
            return view('frontend.pages.booking.package', compact('title', 'package_group_prices', 'package', 'group_price', 'discount', 'pageContent'));
        } catch (\Throwable $e) {
            return abort(404);
        }
    }

    public function bookingPackageSave(Request $request)
    {
        try {
            $validator = Validator::make($request->all(), [
                'package_sub_price_id' => ['required', 'exists:App\Models\PackageSubPrice,id'],
                'arrival_date_time' => ['required'],
                'first_name' => ['required'], 'last_name' => ['required'],
                'email' => ['required', 'email']
            ], [
                'package_sub_price_id.required' => 'Package group price is required'
            ]);
            if ($validator->fails()) {
                return redirect()
                    ->back()
                    ->withErrors($validator)
                    ->withInput();
            }
            $arrival_date_time = Carbon::createFromFormat('m-d-Y h:i A', $request->arrival_date_time)->format('Y-m-d H:i:s');
            $data = [
                'package_id' => $request->hotel_id,
                'user_id' => Auth::id(),
                'discount_id' => $request->discount_id,
                'package_sub_price_id' => $request->package_sub_price_id,
                'arrival_date_time' => $arrival_date_time,
                'first_name' => $request->first_name,
                'last_name' => $request->last_name,
                'email' => $request->email,
                'status' => BOOKING_STATUS_PENDING
            ];

            $this->package_booking->store($data);
            return $this->backWithSuccess(__('Booking done. We send a email in this email account. Please check you email and confirm the confirmation link.'), 'guest.package.order');
        } catch (\Throwable $th) {
            throw $th;
            return abort(404);
        }
    }

    public function confrimBooking(Request $request)
    {
        $error_msg = '';
        try {
            $title = "Confirm Booking";

            if (!Auth::check() || (Auth::check() && Auth::user()->user_type == ADMIN_USER)) {
                Cookie::queue(COOKIE_LOGIN_AFTER_URL, URL::full(), 60);
                return $this->backWithWarning(__('Please login before complete booking'), 'guest.login');
            }

            $methods = (new PaymentMethodService)->getMethods();
            $stripe = $methods->where('method_type', STRIPE_PAYMENT)->first();
            $paypal = $methods->where('method_type', PAYPAL_PAYMENT)->first();

            $stripeService = new StripeService($stripe->secret_key);
            $intent = $stripeService->createIntent();

            if (bookignType($request->type)) {
                if ($request->type == BOOKING_HOTEL) {
                    $booking = $this->hotel_booking->getData()->with(['getCountry', 'getCity'])->where(['id' => decrypt($request->token), 'email' => $request->email, 'user_id' => Auth::user()->id])->first();
                    if (!isset($booking)) {
                        $error_msg = __('Something went wrong');
                        return abort(404, $error_msg);
                    }

                    return view('frontend.pages.booking.payment.hotel', compact('booking', 'title', 'methods', 'intent', 'paypal', 'stripe'));
                } elseif ($request->type == BOOKING_PACKAGE) {
                    $booking = $this->package_booking->getData()->where(['id' => decrypt($request->token), 'email' => $request->email, 'user_id' => Auth::user()->id])->first();
                    if (!isset($booking)) {
                        $error_msg = __('Something went wrong');
                        return abort(404, $error_msg);
                    }

                    return view('frontend.pages.booking.payment.package', compact('booking', 'title', 'methods', 'intent', 'paypal', 'stripe'));
                }
            } else {
                $error_msg = __('This link is not valid');
                return abort(404);
            }
        } catch (\Throwable $th) {
            return $th;
            return abort(404, $error_msg);
        }
    }
}
